<?php
/**
 * Zakra svg icons class
 *
 * @package Zakra
 *
 * @since 3.0.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit();

if ( ! class_exists( 'Zakra_SVG_Icons' ) ) {

	/**
	 * Zakra_SVG_Icons class.
	 */
	class Zakra_SVG_Icons {

		/**
		 * Allowed HTML.
		 *
		 * @var bool[][]
		 */
		public static $allowed_html = array(
			'svg'     => array(
				'class'       => true,
				'xmlns'       => true,
				'width'       => true,
				'height'      => true,
				'viewbox'     => true,
				'aria-hidden' => true,
				'role'        => true,
				'focusable'   => true,
				'fill'        => true,
				'stroke'      => true,
				'fill-rule'   => true,
			),
			'path'    => array(
				'fill'      => true,
				'fill-rule' => true,
				'clip-rule' => true,
				'd'         => true,
				'transform' => true,
			),
			'circle'  => array(
				'cx' => true,
				'cy' => true,
				'r'  => true,
			),
			'polygon' => array(
				'fill'      => true,
				'fill-rule' => true,
				'points'    => true,
				'transform' => true,
				'focusable' => true,
			),
			'rect'    => array(
				'x'      => true,
				'rx'     => true,
				'fill'   => true,
				'height' => true,
				'width'  => true,
			),
		);

		/**
		 * SVG icons.
		 *
		 * @var string[]
		 */
		public static $icons = array(
			'arrow-right-long'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M21.92 12.38a1 1 0 0 0 0-.76 1 1 0 0 0-.21-.33L17.42 7A1 1 0 0 0 16 8.42L18.59 11H2.94a1 1 0 1 0 0 2h15.65L16 15.58A1 1 0 0 0 16 17a1 1 0 0 0 1.41 0l4.29-4.28a1 1 0 0 0 .22-.34Z"/></svg>',
			'arrow-left-long'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M2 12.38a1 1 0 0 1 0-.76.91.91 0 0 1 .22-.33L6.52 7a1 1 0 0 1 1.42 0 1 1 0 0 1 0 1.41L5.36 11H21a1 1 0 0 1 0 2H5.36l2.58 2.58a1 1 0 0 1 0 1.41 1 1 0 0 1-.71.3 1 1 0 0 1-.71-.3l-4.28-4.28a.91.91 0 0 1-.24-.33Z"></path></svg>',
			'chevron-down'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 17.5c-.3 0-.5-.1-.7-.3l-9-9c-.4-.4-.4-1 0-1.4s1-.4 1.4 0l8.3 8.3 8.3-8.3c.4-.4 1-.4 1.4 0s.4 1 0 1.4l-9 9c-.2.2-.4.3-.7.3z"/></svg>',
			'chevron-up'            => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M21 17.5c-.3 0-.5-.1-.7-.3L12 8.9l-8.3 8.3c-.4.4-1 .4-1.4 0s-.4-1 0-1.4l9-9c.4-.4 1-.4 1.4 0l9 9c.4.4.4 1 0 1.4-.2.2-.4.3-.7.3z"/></svg>',
			'magnifying-glass'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M21 22c-.3 0-.5-.1-.7-.3L16.6 18c-1.5 1.2-3.5 2-5.6 2-5 0-9-4-9-9s4-9 9-9 9 4 9 9c0 2.1-.7 4.1-2 5.6l3.7 3.7c.4.4.4 1 0 1.4-.2.2-.4.3-.7.3zM11 4c-3.9 0-7 3.1-7 7s3.1 7 7 7c1.9 0 3.6-.8 4.9-2 0 0 0-.1.1-.1s0 0 .1-.1c1.2-1.3 2-3 2-4.9C18 7.1 14.9 4 11 4z"/></svg>',
			'magnifying-glass-bars' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M17 20H3a1 1 0 0 1 0-2h14a1 1 0 0 1 0 2Zm4-2a1 1 0 0 1-.71-.29L18 15.4a6.29 6.29 0 0 1-10-5A6.43 6.43 0 0 1 14.3 4a6.31 6.31 0 0 1 6.3 6.3 6.22 6.22 0 0 1-1.2 3.7l2.31 2.3a1 1 0 0 1 0 1.42A1 1 0 0 1 21 18ZM14.3 6a4.41 4.41 0 0 0-4.3 4.4 4.25 4.25 0 0 0 4.3 4.2 4.36 4.36 0 0 0 4.3-4.3A4.36 4.36 0 0 0 14.3 6ZM6 14H3a1 1 0 0 1 0-2h3a1 1 0 0 1 0 2Zm0-6H3a1 1 0 0 1 0-2h3a1 1 0 0 1 0 2Z"/></svg>',
			'x-mark'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="m14 12 7.6-7.6c.6-.6.6-1.5 0-2-.6-.6-1.5-.6-2 0L12 10 4.4 2.4c-.6-.6-1.5-.6-2 0s-.6 1.5 0 2L10 12l-7.6 7.6c-.6.6-.6 1.5 0 2 .3.3.6.4 1 .4s.7-.1 1-.4L12 14l7.6 7.6c.3.3.6.4 1 .4s.7-.1 1-.4c.6-.6.6-1.5 0-2L14 12z"/></svg>',
			'ellipsis-vertical'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12.5 3.125C13.5417 3.125 14.375 3.95833 14.375 5C14.375 6.04167 13.5417 6.875 12.5 6.875C11.4583 6.875 10.625 6.04167 10.625 5C10.625 3.95833 11.4583 3.125 12.5 3.125ZM12.5 10.625C13.5417 10.625 14.375 11.4583 14.375 12.5C14.375 13.5417 13.5417 14.375 12.5 14.375C11.4583 14.375 10.625 13.5417 10.625 12.5C10.625 11.4583 11.4583 10.625 12.5 10.625ZM12.5 18.125C13.5417 18.125 14.375 18.9583 14.375 20C14.375 21.0417 13.5417 21.875 12.5 21.875C11.4583 21.875 10.625 21.0417 10.625 20C10.625 18.9583 11.4583 18.125 12.5 18.125Z" stroke="#27272A" stroke-width="2" stroke-miterlimit="10"/></svg>',
			'no-result'             => '<svg width="232" height="186" viewBox="0 0 232 186" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="1.63379" y="1" width="228.866" height="184" rx="7" fill="#F4F4F5" stroke="#3F3F46" stroke-width="2"></rect><circle cx="11.5773" cy="14.751" r="4" fill="#3F3F46"></circle><circle cx="23.5773" cy="14.751" r="4" fill="#3F3F46"></circle><circle cx="35.5773" cy="14.751" r="4" fill="#3F3F46"></circle><rect width="175.751" height="2.95238" rx="1.47619" transform="matrix(1 0 0 -1 48.7444 15.7461)" fill="#3F3F46"></rect><line x1="2.36975" y1="28.5527" x2="229.764" y2="28.5527" stroke="#3F3F46" stroke-width="2"></line><path d="M113 126.076C128.464 126.076 141 113.511 141 98.011C141 82.5111 128.464 69.946 113 69.946C97.536 69.946 85 82.5111 85 98.011C85 113.511 97.536 126.076 113 126.076Z" stroke="#3F3F46" stroke-width="6" stroke-linecap="round" stroke-linejoin="round"></path><path d="M148 133.092L132.775 117.832" stroke="#3F3F46" stroke-width="6" stroke-linecap="round" stroke-linejoin="round"></path></svg>',
			'user'                  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 16 16"><path d="M4.66667 4.66659C4.66667 2.79992 6.13333 1.33325 8 1.33325C9.86667 1.33325 11.3333 2.79992 11.3333 4.66659C11.3333 6.53325 9.86667 7.99992 8 7.99992C6.13333 7.99992 4.66667 6.53325 4.66667 4.66659ZM10.6667 9.33325H5.33333C3.46667 9.33325 2 10.7999 2 12.6666V13.9999C2 14.3999 2.26667 14.6666 2.66667 14.6666H13.3333C13.7333 14.6666 14 14.3999 14 13.9999V12.6666C14 10.7999 12.5333 9.33325 10.6667 9.33325Z"></path></svg>',
			'calendar'              => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 16 16"><path d="M14.0667 4.39992V5.46659C14.0667 5.86659 13.8 6.13325 13.4 6.13325H2.60002C2.20002 6.13325 1.93335 5.86659 1.93335 5.46659V4.39992C1.93335 3.39992 2.80002 2.53325 3.80002 2.53325H4.93335V1.99992C4.93335 1.59992 5.20002 1.33325 5.60002 1.33325C6.00002 1.33325 6.26668 1.59992 6.26668 1.99992V2.53325H9.73335V1.99992C9.73335 1.59992 10 1.33325 10.4 1.33325C10.8 1.33325 11.0667 1.59992 11.0667 1.99992V2.53325H12.2C13.2 2.53325 14.0667 3.39992 14.0667 4.39992ZM13.4 7.46659H2.60002C2.20002 7.46659 1.93335 7.73325 1.93335 8.13325V12.7999C1.93335 13.7999 2.80002 14.6666 3.80002 14.6666H12.2C13.2 14.6666 14.0667 13.7999 14.0667 12.7999V8.13325C14.0667 7.73325 13.8 7.46659 13.4 7.46659Z"></path></svg>',
			'comment'               => '<svg xmlns="http://www.w3.org/2000/svg"viewBox="0 0 16 16"><path d="M14.6667 3.33325V9.99992C14.6667 11.1333 13.8 11.9999 12.6667 11.9999H4.93337L2.46671 14.4666C2.33337 14.5999 2.20004 14.6666 2.00004 14.6666C1.93337 14.6666 1.80004 14.6666 1.73337 14.5999C1.46671 14.5333 1.33337 14.2666 1.33337 13.9999V3.33325C1.33337 2.19992 2.20004 1.33325 3.33337 1.33325H12.6667C13.8 1.33325 14.6667 2.19992 14.6667 3.33325Z"></path></svg>',
			'chevron-right'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M15.707 11.293a1 1 0 010 1.414l-6 6a1 1 0 01-1.414-1.414L13.586 12 8.293 6.707a1 1 0 011.414-1.414l6 6z" /></svg>',
			'chevron-left'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M8.293 12.707a1 1 0 010-1.414l6-6a1 1 0 111.414 1.414L10.414 12l5.293 5.293a1 1 0 01-1.414 1.414l-6-6z"/></svg>',
			'cart'                  => '<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve" viewBox="0 0 24 24"><path d="M18.5 22c-1 0-1.8-.8-1.8-1.8s.8-1.8 1.8-1.8 1.8.8 1.8 1.8-.8 1.8-1.8 1.8zm0-2c-.2 0-.2 0-.2.2s0 .2.2.2.2 0 .2-.2 0-.2-.2-.2zm-8.9 2c-1 0-1.8-.8-1.8-1.8s.8-1.8 1.8-1.8 1.8.8 1.8 1.8-.8 1.8-1.8 1.8zm0-2c-.2 0-.2 0-.2.2s0 .2.2.2.2 0 .2-.2 0-.2-.2-.2zm8.4-2.9h-7.9c-1.3 0-2.4-.9-2.6-2.2L6.1 8.2v-.1L5.4 4H3c-.6 0-1-.4-1-1s.4-1 1-1h3.3c.5 0 .9.4 1 .8L8 7h12.9c.3 0 .6.1.8.4.2.2.3.5.2.8L20.6 15c-.3 1.3-1.3 2.1-2.6 2.1zM8.3 9l1.2 5.6c.1.4.4.5.6.5H18c.1 0 .5 0 .6-.5L19.7 9H8.3z"/></svg>',
			'search'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M11 4a7 7 0 100 14 7 7 0 000-14zm-9 7a9 9 0 1118 0 9 9 0 01-18 0z"/><path d="M15.943 15.943a1 1 0 011.414 0l4.35 4.35a1 1 0 01-1.414 1.414l-4.35-4.35a1 1 0 010-1.414z"/></svg>',
			'calender'              => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M17 2C17 1.44772 16.5523 1 16 1C15.4477 1 15 1.44772 15 2V3H9V2C9 1.44772 8.55228 1 8 1C7.44772 1 7 1.44772 7 2V3H5C3.34315 3 2 4.34315 2 6V10V20C2 21.6569 3.34315 23 5 23H19C20.6569 23 22 21.6569 22 20V10V6C22 4.34315 20.6569 3 19 3H17V2ZM20 9V6C20 5.44772 19.5523 5 19 5H17V6C17 6.55228 16.5523 7 16 7C15.4477 7 15 6.55228 15 6V5H9V6C9 6.55228 8.55228 7 8 7C7.44772 7 7 6.55228 7 6V5H5C4.44772 5 4 5.44772 4 6V9H20ZM4 11H20V20C20 20.5523 19.5523 21 19 21H5C4.44772 21 4 20.5523 4 20V11Z" /></svg>',
			'edit'                  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M17.207 4.207a1.121 1.121 0 011.586 1.586L6.489 18.097l-2.115.529.529-2.115L17.207 4.207zM18 1.88c-.828 0-1.622.329-2.207.914l-12.5 12.5a1 1 0 00-.263.464l-1 4a1 1 0 001.213 1.213l4-1a1 1 0 00.464-.263l12.5-12.5a3.12 3.12 0 00-1.012-5.09A3.121 3.121 0 0018 1.878zM12 19a1 1 0 100 2h9a1 1 0 100-2h-9z" /></svg>',
			'arrow-right'           => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M4 12a1 1 0 011-1h14a1 1 0 110 2H5a1 1 0 01-1-1z" clip-rule="evenodd"/><path d="M11.293 4.293a1 1 0 011.414 0l7 7a1 1 0 010 1.414l-7 7a1 1 0 01-1.414-1.414L17.586 12l-6.293-6.293a1 1 0 010-1.414z" clip-rule="evenodd"/></svg>',
			'arrow-down'            => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" clip-rule="evenodd" d="M5.293 8.293a1 1 0 011.414 0L12 13.586l5.293-5.293a1 1 0 111.414 1.414l-6 6a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414z"/></svg>',
			'arrow-up'              => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M12 20a1 1 0 01-1-1V5a1 1 0 112 0v14a1 1 0 01-1 1z"></path><path fill-rule="evenodd" d="M4.293 12.707a1 1 0 010-1.414l7-7a1 1 0 011.414 0l7 7a1 1 0 01-1.414 1.414L12 6.414l-6.293 6.293a1 1 0 01-1.414 0z"></path></svg>',
			'filter'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path clip-rule="evenodd" d="M4 2a1 1 0 011 1v7a1 1 0 11-2 0V3a1 1 0 011-1zm1 13h2a1 1 0 100-2H1a1 1 0 100 2h2v6a1 1 0 102 0v-6zm8-3a1 1 0 10-2 0v9a1 1 0 102 0v-9zM12 2a1 1 0 011 1v4h2a1 1 0 110 2H9a1 1 0 110-2h2V3a1 1 0 011-1zm8 13h3a1 1 0 110 2h-2v4a1 1 0 11-2 0v-4h-2a1 1 0 110-2h3zm0-13a1 1 0 011 1v9a1 1 0 11-2 0V3a1 1 0 011-1z"/></svg>',
			'tag'                   => '<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve" viewBox="0 0 24 24"><path d="M14.4 21.5c-.3 0-.5-.1-.7-.3-.4-.4-.4-1 0-1.4l5.2-5.2c.7-.7 1.1-1.7 1.1-2.7 0-1-.4-2-1.1-2.7l-4.9-5c-.4-.4-.4-1 0-1.4.4-.4 1-.4 1.4 0l4.9 4.9c1.1 1.1 1.7 2.6 1.7 4.1s-.6 3-1.7 4.1l-5.2 5.2c-.2.3-.5.4-.7.4zm-1-2.7 4.7-4.7c1.3-1.3 1.3-3.3 0-4.6l-5.9-5.9c-.6-.6-1.4-.9-2.3-.9H4.7C3.2 2.7 2 3.9 2 5.4v5.2c0 .9.3 1.7.9 2.3l5.9 5.9c.6.6 1.5.9 2.3.9.8 0 1.6-.3 2.3-.9zM9.9 4.7c.3 0 .6.1.9.4l5.9 5.9c.5.5.5 1.3 0 1.8L12 17.3c-.5.5-1.3.5-1.8 0l-5.9-5.9c-.2-.2-.3-.5-.3-.8V5.4c0-.4.3-.7.7-.7h5.2zm-3 4c-.6 0-1.1-.5-1.1-1.1s.5-1.1 1.1-1.1S8 7 8 7.6s-.5 1.1-1.1 1.1z"></path></svg>',
			'close'                 => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M5.293 5.293a1 1 0 011.414 0L12 10.586l5.293-5.293a1 1 0 111.414 1.414L13.414 12l5.293 5.293a1 1 0 01-1.414 1.414L12 13.414l-5.293 5.293a1 1 0 01-1.414-1.414L10.586 12 5.293 6.707a1 1 0 010-1.414z" /></svg>',
			'grid'                  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M3 2a1 1 0 00-1 1v7a1 1 0 001 1h7a1 1 0 001-1V3a1 1 0 00-1-1H3zm1 7V4h5v5H4zM14 2a1 1 0 00-1 1v7a1 1 0 001 1h7a1 1 0 001-1V3a1 1 0 00-1-1h-7zm1 7V4h5v5h-5zM13 14a1 1 0 011-1h7a1 1 0 011 1v7a1 1 0 01-1 1h-7a1 1 0 01-1-1v-7zm2 1v5h5v-5h-5zM3 13a1 1 0 00-1 1v7a1 1 0 001 1h7a1 1 0 001-1v-7a1 1 0 00-1-1H3zm1 7v-5h5v5H4z"/></svg>',
			'close-circle'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 3a9 9 0 100 18 9 9 0 000-18zM1 12C1 5.925 5.925 1 12 1s11 4.925 11 11-4.925 11-11 11S1 18.075 1 12z" /><path d="M15.707 8.293a1 1 0 010 1.414l-6 6a1 1 0 01-1.414-1.414l6-6a1 1 0 011.414 0z" /><path d="M8.293 8.293a1 1 0 011.414 0l6 6a1 1 0 01-1.414 1.414l-6-6a1 1 0 010-1.414z" /></svg>',
			'bars-sort'             => '<svg xmlns="http://www.w3.org/2000/svg" data-name="Layer 1" viewBox="0 0 24 24"><path d="M9 19H3a1 1 0 0 1 0-2h6a1 1 0 0 1 0 2Zm6-6H3a1 1 0 0 1 0-2h12a1 1 0 0 1 0 2Zm6-6H3a.94.94 0 0 1-1-1 .94.94 0 0 1 1-1h18a.94.94 0 0 1 1 1 .94.94 0 0 1-1 1Z"></path></svg>',
			'folder'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M19.2 5.60039H11.6L10.1 3.30039C9.9 3.10039 9.6 2.90039 9.3 2.90039H4.8C3.3 2.90039 2 4.20039 2 5.70039V18.3004C2 19.8004 3.3 21.1004 4.8 21.1004H19.2C20.7 21.1004 22 19.8004 22 18.3004V8.40039C22 6.90039 20.7 5.60039 19.2 5.60039Z" ></path></svg>',
			'bars'                  => '<svg xmlns="http://www.w3.org/2000/svg" data-name="Layer 1" viewBox="0 0 24 24"><path d="M21 19H3a1 1 0 0 1 0-2h18a1 1 0 0 1 0 2Zm0-6H3a1 1 0 0 1 0-2h18a1 1 0 0 1 0 2Zm0-6H3a1 1 0 0 1 0-2h18a1 1 0 0 1 0 2Z"/></svg>',
			'heart'                 => '<svg xmlns="http://www.w3.org/2000/svg" data-name="Layer 1" viewBox="0 0 24 24"><path d="M12 20.85a1 1 0 0 1-.71-.29l-7.61-7.62a5.73 5.73 0 1 1 8.11-8.11L12 5l.21-.21a5.73 5.73 0 0 1 8.11 0 5.73 5.73 0 0 1 0 8.11l-7.61 7.62a1 1 0 0 1-.71.33ZM7.74 5.15a3.74 3.74 0 0 0-2.65 6.38L12 18.44l6.91-6.91a3.74 3.74 0 0 0 0-5.28 3.75 3.75 0 0 0-5.29 0l-.91.92a1 1 0 0 1-1.42 0l-.91-.91a3.7 3.7 0 0 0-2.64-1.11Z"/></svg>',
			'cart-1'                => '<svg class="zak-icon zakra-icon--cart" xmlns="http://www.w3.org/2000/svg" xml:space="preserve" viewBox="0 0 24 24"><path d="M18.5 22c-1 0-1.8-.8-1.8-1.8s.8-1.8 1.8-1.8 1.8.8 1.8 1.8-.8 1.8-1.8 1.8zm0-2c-.2 0-.2 0-.2.2s0 .2.2.2.2 0 .2-.2 0-.2-.2-.2zm-8.9 2c-1 0-1.8-.8-1.8-1.8s.8-1.8 1.8-1.8 1.8.8 1.8 1.8-.8 1.8-1.8 1.8zm0-2c-.2 0-.2 0-.2.2s0 .2.2.2.2 0 .2-.2 0-.2-.2-.2zm8.4-2.9h-7.9c-1.3 0-2.4-.9-2.6-2.2L6.1 8.2v-.1L5.4 4H3c-.6 0-1-.4-1-1s.4-1 1-1h3.3c.5 0 .9.4 1 .8L8 7h12.9c.3 0 .6.1.8.4.2.2.3.5.2.8L20.6 15c-.3 1.3-1.3 2.1-2.6 2.1zM8.3 9l1.2 5.6c.1.4.4.5.6.5H18c.1 0 .5 0 .6-.5L19.7 9H8.3z"/></svg>',
			'cart-2'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M15.083 10.255a.75.75 0 0 1 .667.751l-.005.077-1 9a.75.75 0 1 1-1.49-.166l1-9 .012-.075a.75.75 0 0 1 .816-.587Zm-.455-6.906a.75.75 0 0 1 .981.215l.042.064 4 7 .035.068a.75.75 0 0 1-1.295.74l-.042-.064-4-7-.034-.068a.75.75 0 0 1 .313-.955Z"/><path d="M22 10.25a.75.75 0 0 1 0 1.5H2a.75.75 0 1 1 0-1.5h20Z"/><path d="M19.87 10.832a.75.75 0 1 1 1.461.336l-1.7 7.4h-.001a2.748 2.748 0 0 1-2.73 2.181l-9.8.001a2.75 2.75 0 0 1-2.735-2.2l-1.598-7.392-.012-.075a.75.75 0 0 1 1.458-.315l.02.074 1.6 7.4.002.008.026.107a1.25 1.25 0 0 0 1.224.893h9.83a1.251 1.251 0 0 0 1.25-1l.004-.017 1.7-7.4Z"/><path d="M19.5 14.75a.75.75 0 0 1 0 1.5h-15a.75.75 0 1 1 0-1.5h15ZM8.39 3.564a.75.75 0 0 1 1.296.74l-.035.068-4 7a.75.75 0 0 1-1.302-.744l4-7 .042-.064Z"/><path d="M8.917 10.255a.75.75 0 0 1 .816.587l.012.075 1 9 .005.077a.75.75 0 0 1-1.483.164l-.012-.075-1-9-.005-.077a.75.75 0 0 1 .667-.751Z"/></svg>',
			'cart-3'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M15 6a3 3 0 0 0-6 0v.75h6V6ZM5.444 8.256a.378.378 0 0 0-.29.263l-.014.067-1.264 12a.377.377 0 0 0 .373.414h15.5a.374.374 0 0 0 .374-.333v-.081l-1.263-12a.376.376 0 0 0-.373-.336H16.5v1.412l.046.042.074.083a1.124 1.124 0 1 1-1.99.824l-.005-.111.006-.111c.025-.258.14-.5.324-.685l.045-.04V8.25H9v1.412l.046.042.074.083a1.125 1.125 0 1 1-1.99.824l-.005-.111.006-.111c.025-.258.14-.5.324-.685l.045-.04V8.25H5.513l-.069.006ZM16.5 6.75h1.987c.96 0 1.765.725 1.865 1.679l1.263 12 .01.206A1.875 1.875 0 0 1 19.75 22.5H4.25a1.878 1.878 0 0 1-1.394-.62 1.876 1.876 0 0 1-.471-1.451l1.263-12 .027-.171A1.874 1.874 0 0 1 5.514 6.75H7.5V6a4.5 4.5 0 0 1 9 0v.75Z"/></svg>',
			'cart-4'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M20.25 6.249 17.625 2.75H6.375L3.75 6.249V20A1.25 1.25 0 0 0 5 21.25h14A1.25 1.25 0 0 0 20.25 20V6.249ZM21.75 20A2.75 2.75 0 0 1 19 22.75H5A2.75 2.75 0 0 1 2.25 20V6l.01-.12a.751.751 0 0 1 .14-.33l3-4 .057-.068A.75.75 0 0 1 6 1.25h12a.75.75 0 0 1 .6.3l3 4a.75.75 0 0 1 .15.45v14Z"/><path d="M21 5.25a.75.75 0 0 1 0 1.5H3a.75.75 0 0 1 0-1.5h18ZM7.25 10a.75.75 0 0 1 1.5 0A3.25 3.25 0 0 0 12 13.25l.161-.004A3.25 3.25 0 0 0 15.25 10a.75.75 0 0 1 1.5 0 4.75 4.75 0 0 1-4.515 4.744L12 14.75A4.75 4.75 0 0 1 7.25 10Z"/></svg>',
			'cart-5'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M3.75 9v10.125c0 1.003.872 1.875 1.875 1.875h12.75c1.025 0 1.875-.835 1.875-1.816V9H3.75Zm12-2.25a3.75 3.75 0 0 0-7.5 0v.75h7.5v-.75Zm1.5.75h3a1.5 1.5 0 0 1 1.5 1.5v10.184c0 1.853-1.565 3.316-3.375 3.316H5.625c-1.832 0-3.375-1.543-3.375-3.375V9a1.5 1.5 0 0 1 1.5-1.5h3v-.75a5.25 5.25 0 0 1 10.5 0v.75Z"/></svg>',
			'cart-6'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 2.001a7.61 7.61 0 0 1 5.37 2.213l.258.27a7.554 7.554 0 0 1 1.933 4.347c.266.027.527.091.774.193l.14.063.203.108a2.691 2.691 0 0 1 1.245 2.953v.001l-1.565 6.407-.006.02a4.65 4.65 0 0 1-1.52 2.352l-.149.119a4.674 4.674 0 0 1-2.83.953H8.145a4.675 4.675 0 0 1-2.676-.84l-.154-.113A4.65 4.65 0 0 1 3.7 18.76l-.055-.184-.006-.02-1.564-6.409a2.691 2.691 0 0 1 .502-2.302l.1-.117a2.71 2.71 0 0 1 1.76-.898A7.55 7.55 0 0 1 6.63 4.214l.274-.259A7.61 7.61 0 0 1 12 2v.001Zm-7.433 8.323a1.214 1.214 0 0 0-.721.349l-.09.102a1.192 1.192 0 0 0-.223 1.017l1.562 6.398c.184.663.58 1.247 1.128 1.663l.213.148c.508.324 1.1.499 1.709.499h7.708c.695 0 1.37-.227 1.922-.646l.199-.165a3.15 3.15 0 0 0 .932-1.51l1.56-6.386.02-.1a1.191 1.191 0 0 0-.66-1.255v.002a1.25 1.25 0 0 0-.396-.116l-.139-.008H4.702l-.135.008Zm2.77 7.031V13.46a.75.75 0 0 1 1.5 0v3.895a.75.75 0 0 1-1.5 0Zm3.913 0V13.46a.75.75 0 0 1 1.5 0v3.895a.75.75 0 0 1-1.5 0Zm3.913 0V13.46a.75.75 0 0 1 1.5 0v3.895a.75.75 0 0 1-1.5 0ZM11.698 3.507A6.11 6.11 0 0 0 7.907 5.07l-.22.207a6.051 6.051 0 0 0-1.735 3.54h12.097a6.045 6.045 0 0 0-1.736-3.539A6.11 6.11 0 0 0 12 3.5l-.302.007Z"/></svg>',
			'cart-7'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M6.75 19.5a1.5 1.5 0 1 1 3 0 1.5 1.5 0 0 1-3 0Zm10.5 0a1.5 1.5 0 1 1 3 0 1.5 1.5 0 0 1-3 0ZM5.25 3l.134.012a.75.75 0 0 1 .604.608l2.14 12.13H19.5a.75.75 0 0 1 0 1.5h-12a.75.75 0 0 1-.738-.62L4.622 4.5H2.25a.75.75 0 0 1 0-1.5h3Z"/><path d="M20.543 5.25c.125 0 .248.02.365.06l.115.047.11.06c.07.043.136.095.195.152l.084.092.074.1c.046.07.083.145.112.222l.036.12.023.122c.011.082.013.165.006.248l-.018.123-1.35 6.75a1.126 1.126 0 0 1-1.006.9l-.097.004H7.5a.75.75 0 0 1 0-1.5h11.386l1.199-6H6a.75.75 0 0 1 0-1.5h14.543Z"/></svg>',
			'cart-8'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12.36 3.25c1.57 0 2.815-.001 3.805.121 1.01.125 1.842.387 2.55.974.707.587 1.117 1.357 1.426 2.327.303.95.531 2.173.82 3.715l.287 1.544c.265 1.46.457 2.672.502 3.68.06 1.365-.144 2.495-.914 3.423-.77.927-1.844 1.335-3.196 1.527-1.332.19-3.07.189-5.28.189h-.72c-2.21 0-3.949.002-5.28-.189-1.268-.18-2.29-.55-3.047-1.359l-.148-.168c-.77-.928-.973-2.058-.914-3.423.06-1.344.381-3.052.789-5.224l.208-1.097c.2-1.039.384-1.906.61-2.618.31-.97.72-1.74 1.428-2.327.707-.587 1.54-.85 2.55-.974.989-.122 2.235-.121 3.803-.121h.722Zm-.72 1.5c-1.607 0-2.744.001-3.62.11-.856.105-1.373.304-1.776.639-.404.335-.694.807-.955 1.628-.201.63-.37 1.416-.567 2.436l-.208 1.1c-.416 2.218-.71 3.802-.764 5.014-.052 1.191.139 1.881.569 2.4l.172.184c.427.41 1.048.669 2.08.816 1.202.171 2.813.173 5.07.173h.719c2.256 0 3.867-.002 5.068-.173 1.18-.168 1.824-.483 2.254-1l.15-.204c.324-.495.464-1.154.418-2.196-.04-.91-.215-2.027-.476-3.465l-.287-1.549c-.296-1.578-.507-2.695-.775-3.536-.261-.821-.552-1.293-.956-1.628-.403-.335-.92-.534-1.776-.64-.876-.108-2.013-.109-3.62-.109h-.72Z"/><path d="M14.151 7.68a.75.75 0 0 1 1.408.496l-.022.074a3.752 3.752 0 0 1-6.968.27l-.106-.27-.023-.074a.75.75 0 0 1 1.408-.496l.029.07.064.162a2.251 2.251 0 0 0 4.182-.162l.028-.07Z"/></svg>',
			'cart-9'                => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M14.465 11.849a.751.751 0 0 1 1.166.932l-.045.063-4.2 5.25a.75.75 0 0 1-1.098.079l-.058-.06-1.8-2.104 1.14-.976 1.21 1.414 3.634-4.54.05-.058ZM8.513 14.95a.75.75 0 0 1 1.057.082l-1.14.976a.75.75 0 0 1 .083-1.058Z"/><path d="M3.75 9v10.125c0 1.003.872 1.875 1.875 1.875h12.75c1.025 0 1.875-.835 1.875-1.816V9H3.75Zm12-2.25a3.75 3.75 0 0 0-7.5 0v.75h7.5v-.75Zm1.5.75h3a1.5 1.5 0 0 1 1.5 1.5v10.184c0 1.853-1.565 3.316-3.375 3.316H5.625c-1.832 0-3.375-1.543-3.375-3.375V9a1.5 1.5 0 0 1 1.5-1.5h3v-.75a5.25 5.25 0 0 1 10.5 0v.75Z"/></svg>',
			'cart-10'               => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M8.34 19.5a.84.84 0 0 0-.757-.837L7.5 18.66a.841.841 0 0 0-.84.841l.003.083a.84.84 0 0 0 1.678-.083Zm8.997-.083a.84.84 0 0 0-.754-.754l-.083-.004a.841.841 0 0 0-.84.841l.003.083a.84.84 0 0 0 1.678-.083l-.004-.083ZM3.75 3l.139.013a.75.75 0 0 1 .605.644L5.912 15h12.202l1.25-7.5H6V6h14.25a.75.75 0 0 1 .74.873l-1.5 9a.75.75 0 0 1-.74.627H5.25a.75.75 0 0 1-.744-.657L3.088 4.5H1.5V3h2.25ZM15 12v1.5H9V12h6Zm0-3v1.5H9V9h6ZM9.84 19.5a2.34 2.34 0 0 1-4.67.231l-.01-.231a2.34 2.34 0 0 1 2.34-2.34l.231.01a2.34 2.34 0 0 1 2.11 2.33Zm8.99.231a2.34 2.34 0 0 1-4.66 0l-.01-.231a2.34 2.34 0 0 1 2.34-2.34l.231.01a2.34 2.34 0 0 1 2.11 2.33l-.012.231Z"/></svg>',
		);

		/**
		 * Get the SVG icon.
		 *
		 * @param string $icon Default is empty.
		 * @param bool   $echo Default is true.
		 * @param array  $args Default is empty.
		 *
		 * @return string|null
		 */
		public static function get_svg( $icon = '', $echo = true, $args = array() ) {

			$icons = self::get_icons();
			$atts  = '';
			$svg   = '';

			if ( ! empty( $args ) ) {

				foreach ( $args as $key => $value ) {

					if ( ! empty( $value ) ) {

						$atts .= esc_html( $key ) . '="' . esc_attr( $value ) . '" ';
					}
				}
			}

			if ( array_key_exists( $icon, $icons ) ) {

				$repl = sprintf( '<svg class="zak-icon zakra-icon--%1$s" %2$s', $icon, $atts );
				$svg  = preg_replace( '/^<svg /', $repl, trim( $icons[ $icon ] ) );
				$svg  = preg_replace( "/([\n\t]+)/", ' ', $svg );
				$svg  = preg_replace( '/>\s*</', '><', $svg );
			}

			if ( ! $svg ) {

				return null;
			}

			if ( $echo ) {

				echo wp_kses( $svg, self::$allowed_html );
			} else {

				return wp_kses( $svg, self::$allowed_html );
			}
		}

		/**
		 * Get all SVG icons.
		 *
		 * @return mixed|void
		 */
		public static function get_icons() {

			/**
			 * Filter for svg icons.
			 *
			 * @since 3.0.0
			 */
			return apply_filters( 'zakra_svg_icons', self::$icons );
		}
	}
}
